/*------------------------------------------------------------------------
  **** BEGIN LICENSE BLOCK ****
  Version: MPL 1.1/GPL 2.0/LGPL 2.1
  Copyright(c) 2004-2010 Created by Eric Fredericksen (www.pttpsystems.com)
  All Rights Reserved.

  This program are subject to the Mozilla Public License Version
  1.1 (the "License"); you may not use this file except in compliance with
  the License. You may obtain a copy of the License at
  http://www.mozilla.org/MPL/

  This program is free software. Software distributed under the License
  is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND,
  either express or implied. See the License for the specific language
  governing rights and limitations under the License.

  Alternatively, the contents of this file may be used under the terms of
  either the GNU General Public License Version 2 or later (the "GPL"), or
  the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
  in which case the provisions of the GPL or the LGPL are applicable instead
  of those above. If you wish to allow use of your version of this file only
  under the terms of either the GPL or the LGPL, and not to allow others to
  use your version of this file under the terms of the MPL, indicate your
  decision by deleting the provisions above and replace them with the notice
  and other provisions required by the GPL or the LGPL. If you do not delete
  the provisions above, a recipient may use your version of this file under
  the terms of any one of the MPL, the GPL or the LGPL.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA
  **** END LICENSE BLOCK ****
------------------------------------------------------------------------*/

#include "stdafx.h"

#include "ServerTools.h"

//==================================================
//
//==================================================
bool CServer::InitWSA(void)
{
	return( 0 == WSAStartup(MAKEWORD(2, 0), &m_oWSAData) );
}


//==================================================
//
//==================================================
void CServer::SetListenAddress(DWORD _dwHBO_IP, USHORT _HBO_Port)
{
	m_oListenAddress.sin_family			= AF_INET;
	m_oListenAddress.sin_addr.s_addr	= htonl(_dwHBO_IP);
	m_oListenAddress.sin_port			= htons(_HBO_Port);
}

//==================================================
//
//==================================================
CServer::CServer(void)
{
	m_bWSAInitialized = InitWSA();
	m_bSocketInitialized = false;

	SetListenAddress(DEFAULT_LISTEN_IP, DEFAULT_LISTEN_PORT);
}

//==================================================
//
//==================================================
CServer::~CServer()
{
	if( m_bWSAInitialized ) WSACleanup();
}

//==================================================
// IP in host byte order
//==================================================
const char * CServer::DefaultDottedIP(void)
{
	DWORD dwIP = DefaultIP();
	return( DottedIP( dwIP ) );
}

//==================================================
// IP in host byte order
//==================================================
const char * CServer::DottedIP(DWORD _HBO_IP)
{
	DWORD dwIP = htonl(_HBO_IP);
	return( inet_ntoa( *(in_addr*)&dwIP ) );
}

//==================================================
// IP in host byte order
//==================================================
DWORD CServer::DefaultIP(void)
{

	DWORD dwIP = INADDR_NONE;
	if( m_bWSAInitialized )
	{
		hostent * poHostName = gethostbyname(NULL);
		if( poHostName )
		{
			dwIP = *((DWORD*)poHostName->h_addr);
		}
	}
	return( ntohl(dwIP) );
}

void CServer::PrintWSALastError(void)
{
	DWORD dwError = WSAGetLastError();
	fprintf(stderr, _T("!!WSALastError(%u:%X:%d)\n"), dwError,dwError,dwError  );
}


//==================================================
//
//==================================================
bool CServer::BindTo(DWORD _dwHBO_IP, USHORT _HBO_Port, USHORT _usMaxWaitingConnections)
{
	if( !m_bWSAInitialized ) return( false );

	SetListenAddress(_dwHBO_IP, _HBO_Port);

	// do all the work of setting up the listening socket

	m_sListenSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if( INVALID_SOCKET  == m_sListenSocket )
	{
		// record the error here for later inspection
		fprintf(stderr, _T("!!Could not create socket\n")  );
		PrintWSALastError();
		return( false );
	}

	// do this so that we don't run out of source ports...
	BOOL bReuse = TRUE;
	int iResult = setsockopt(m_sListenSocket, SOL_SOCKET, SO_REUSEADDR, (LPCSTR)&bReuse, sizeof(bReuse) );
	if( SOCKET_ERROR == iResult )
	{
		// provide feedback/warning on failure for later inspection
		fprintf(stderr, _T("Could not setsockopt(SO_REUSEADDR)\n") );
		PrintWSALastError();
		// not a critical error
	}

	// make sure the timeouts are what we want
	DWORD dwTimeout_ms = DEFAULT_SOCKET_TIMEOUT_ms;
	iResult = setsockopt(m_sListenSocket, SOL_SOCKET, SO_RCVTIMEO, (const char*)&dwTimeout_ms, sizeof(dwTimeout_ms) );
	if( SOCKET_ERROR  == iResult )
	{
		// send warning message
		fprintf(stderr, _T("Could not setsockopt(SO_RCVTIMEO, %d)\n"), dwTimeout_ms );
		PrintWSALastError();
		// not a critical error
	}

	iResult = setsockopt(m_sListenSocket, SOL_SOCKET, SO_SNDTIMEO, (const char*)&dwTimeout_ms, sizeof(dwTimeout_ms) );
	if( SOCKET_ERROR  == iResult )
	{
		// send warning message
		fprintf(stderr, _T("Could not setsockopt(SO_SNDTIMEO, %d)\n"), dwTimeout_ms );
		PrintWSALastError();
		// not a critical error
	}

	iResult = bind(m_sListenSocket, (sockaddr*)&m_oListenAddress, sizeof(m_oListenAddress) );
	if( SOCKET_ERROR  == iResult )
	{
		// send warning message
		fprintf(stderr, _T("!!Could not bind(%s:%u)\n"), DottedIP(_dwHBO_IP),  _HBO_Port);
		PrintWSALastError();
	// critical error
		return(false);
	}

	iResult = listen(m_sListenSocket, _usMaxWaitingConnections );
	if( SOCKET_ERROR  == iResult )
	{
		// send warning message
		fprintf(stderr, _T("!!Could not listen(%u:%u)\n"), m_sListenSocket,  _usMaxWaitingConnections);
		PrintWSALastError();
		// critical error
		return(false);
	}

	m_bSocketInitialized = true;
	fprintf(stderr, _T("Server listening on (%s:%u) with queue length %d\n"), 
		DottedIP(DefaultIP()),  
		_HBO_Port,
		_usMaxWaitingConnections
		);

	return( m_bSocketInitialized );
}


//==================================================
//
//==================================================
void CServer::StartListening(void)
{
	if( !m_bSocketInitialized )
	{
		fprintf(stderr,_T("Listening socket not yet initialized\n") );
		return;
	}

	while(TRUE)
	{
		// this does not time out because the socket is blocking.
		SOCKET sNewSocket = accept(m_sListenSocket, NULL, NULL);
		if( SOCKET_ERROR == sNewSocket )
		{
			// record error message
			fprintf(stderr,_T("Error on accept()\n") );
			PrintWSALastError();
			break; // quit from the loop and exit the server
		}
		// hand off the connection 
		DispatchConnection( sNewSocket );
	}
}


